'use client'
import { useEffect, useState } from 'react'
import { Table, Row, Col } from 'react-bootstrap';
import SearchInput from '../SearchInput/SearchInput';

import api from '@/utils/api'
import Image from 'next/image';
import CustomProduct from '@/interfaces/CustomProduct';
import Pagination from '../Pagination/Pagination';
import Link from 'next/link';
import ProductCard from '../ProductCard/ProductCard';

import { useRouter } from 'next/navigation';
import { ToastContainer, toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';


const CustomProductList = ({ type }: { type: string }) => {

    const STORE_ID = process.env.NEXT_PUBLIC_STORE_ID;

    const router = useRouter();

    const [currentPage, setCurrentPage] = useState<number>(1)
    const [pagesOnApi, setPagesOnApi] = useState<number>(0)
    const [limit, setLimit] = useState<number>(10)
    const [offset, setOffset] = useState<number>(1)

    const [searchQuery, setSearchQuery] = useState<string>('')
    const [products, setProducts] = useState<CustomProduct[]>()

    const [selectedProduct, setSelectedProduct] = useState<CustomProduct>()

    const [loading, setLoading] = useState<boolean>(false)

    const getPageTitle = () => {
        switch (type) {
            case 'custom':
                return 'Customização de Produto'
            case 'description':
                return 'Descrição de Produto'
            case 'kit':
                return 'Kit de Produto'
        }

    }

    useEffect(() => {
        try {
            setLoading(true)

            const getProducts = async () => {

                let url = `product/nuvem?nuvemShopStoreId=${STORE_ID}&page=${currentPage}&per_page=${limit}`;

                if (searchQuery) {
                    url = `${url}&q=${searchQuery}`;
                }

                const response = await api.get(url)

                setProducts(response.data[0])
                setPagesOnApi(response.data[1])
                setLoading(false)
            }
            getProducts()
        }
        catch (error) {
            console.log(error)
        }
    }, [searchQuery, currentPage])

    const handlePageChange = (page: number) => {
        setCurrentPage(page)
    }

    const handleContinue = async () => {
        if (!selectedProduct || (selectedProduct === undefined)) return toast.error('Selecione um produto para continuar')

        const { id, name, images } = selectedProduct;

        const data = {
            id: id,
            nome: name.pt,
            foto: images[0].src,
            store_id: 3048218
        }

        try {
            const response = await api.post(`/product`, data)
            console.log(response)

        } catch (error) {
            console.log(error)
        }
        console.log(type)

        if (type === 'custom') {
            const query = new URLSearchParams({
                type: 'custom',
                name: name?.pt || '',
                image: images[0]?.src || ''
            });
            return router.push(`/new_custom_product/${id}?${query.toString()}`);
        }

        if (type === 'description') {
            const query = new URLSearchParams({
                type: 'description',
                name: name?.pt || '',
                image: images[0]?.src || ''
            });
            return router.push(`/new_description_product/${id}?${query.toString()}`);
        }

        if (type === 'kit') {
            const query = new URLSearchParams({
                type: 'kit',
                name: name?.pt || '',
                image: images[0]?.src || ''
            });
            return router.push(`/kit/${id}?${query.toString()}`);
        }
    }

    return (
        <div className='w-80 px-5'>
            <div className='bg-light rounded-4 p-3'>
                <Row>
                    {/* Product list */}
                    <Col lg={7}>
                        <h4 className='fw-bold mb-4'>{getPageTitle()}</h4>
                        <div>
                            <div className='fs-7 mb-3'>
                                Para criar sua customização, selecione um produto cadastrado em sua loja
                            </div>
                            <SearchInput setQuery={setSearchQuery} />
                        </div>
                        <div className='overflow-y-scroll' style={{ height: '300px' }}>
                            <Table className="table">
                                <thead>
                                    <tr>
                                        <th className='text-center'>Produto</th>
                                        <th className='text-center'>Ações</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {loading ? (
                                        <tr>
                                            <td colSpan={2} className='text-center'>
                                                <div className="spinner-border text-primary" role="status">
                                                    <span className="visually-hidden">Loading...</span>
                                                </div>
                                            </td>
                                        </tr>
                                    ) : (
                                        <>
                                            {products && (
                                                <>
                                                    {products.map((product, index) => (
                                                        <tr key={index}>
                                                            <td>
                                                                <img src={product?.images?.[0]?.src} className="me-4" width={90} height={90} alt="" />
                                                                <span className='fs-7'>
                                                                    {product.name.pt}
                                                                </span>
                                                            </td>
                                                            <td className="text-center">
                                                                <div className='d-flex flex-column fs-7'>
                                                                    {selectedProduct?.id === product.id ? (
                                                                        <button onClick={e => setSelectedProduct(undefined)} className='btn text-danger text-decoration-underline'>Remover</button>
                                                                    ) : (
                                                                        <button onClick={e => setSelectedProduct(product)} className='btn text-success-2 text-decoration-underline'>Adicionar</button>
                                                                    )}
                                                                    <Link href={product.canonical_url} target='_blank' className='btn text-primary text-decoration-underline'>Visualizar</Link>
                                                                </div>
                                                            </td>
                                                        </tr>
                                                    ))}
                                                </>
                                            )}
                                        </>
                                    )}
                                </tbody>
                            </Table>
                        </div>
                    </Col>
                    {/* Selected product */}
                    <Col lg={5}>
                        <div>
                            {selectedProduct ? (
                                <ProductCard id={selectedProduct.id} name={selectedProduct.name.pt} image={selectedProduct?.images[0]?.src} />
                            ) : (
                                <div className='d-flex align-items-center justify-content-center shadow-sm bg-secondary bg-opacity-25 rounded-4 fs-7 p-5 mb-3 mt-5'>
                                    <div className="opacity-75">
                                        Nenhum produto selecionado
                                    </div>
                                </div>
                            )}

                            <div className='d-flex justify-content-end'>
                                <Link href='/' className='btn btn-dark fs-7 me-3'>
                                    Voltar
                                </Link>
                                <button onClick={e => handleContinue()} className='btn btn-primary fs-7'>
                                    Continuar
                                    <Image src="/svg/arrow-right.svg" className="ms-2" width={16} height={16} alt="" />
                                </button>
                            </div>
                        </div>
                    </Col>
                </Row>
                <Pagination currentPage={currentPage} onPageChange={handlePageChange} pagesOnApi={pagesOnApi} />
            </div>
            <ToastContainer />
        </div>
    )
}

export default CustomProductList