
import React, { useEffect, useState } from 'react'
import { Row, Col, Tabs, Tab } from 'react-bootstrap';
import ProductCard from '../ProductCard/ProductCard';
import { ToastContainer, toast } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css'
import ImageUploading, { ImageListType } from 'react-images-uploading';
import Image from 'next/image';
import api from '@/utils/api';


interface Props {
  id: number;
  name: string;
  image: string;
}

interface Tab {
  product_id: number;
  id: number | undefined;
  titulo: string;
  TabElements: TabElements[];
}

interface TabElements {
  id?: number;
  tab_id: number;
  tipo: string;
  titulo?: string;
  texto: string;
  link?: string;
}

const NewDescriptionProduct: React.FC<Props> = ({ id, name, image }) => {

  const [tabs, setTabs] = useState<Tab[]>([]);
  const [selectedTab, setSelectedTab] = useState<Tab>();

  const [editMode, setEditMode] = useState<boolean>(false);
  const [tabTitle, setTabTitle] = useState<string>('');

  // simple text
  const [simpleText, setSimpleText] = useState<string>('')
  // images
  const [imageElementIds, setImageElementIds] = useState<number[]>([]);
  const [images, setImages] = useState<ImageListType>([])
  const [imgTitle, setImgTitle] = useState<string[]>([])
  const [imgText, setImgText] = useState<string[]>([])
  const [imgLink, setImgLink] = useState<string[]>([])
  // video
  const [videoTitle, setVideoTitle] = useState<string>('')
  const [videoText, setVideoText] = useState<string>('')
  const [videoLink, setVideoLink] = useState<string>('')

  // fetch existing tabs
  useEffect(() => {
    const getTabs = async () => {
      try {
        const response = await api.get(`/tab?offset=0&limit=5&product_id=${id}`)
        setTabs(response.data[0])
      } catch (error) {
        console.log(error)
      }

    }
    getTabs()
  }, [])

  const handleAddTab = async () => {
    if (tabs.length >= 4) return toast.error('Você atingiu o limite de tabs')


    const data = {
      product_id: id,
      titulo: `Tab ${tabs.length + 1}`,
    }

    try {
      const response = await api.post(`/tab`, data);
      const newTab: Tab = {
        product_id: id,
        id: response.data.id,
        titulo: `Tab ${tabs.length + 1}`,
        TabElements: [],
      };
      console.log(response.data)
      setTabs([...tabs, newTab]);
    } catch (error) {
      console.error('Erro ao criar nova tab', error);
      toast.error('Erro ao criar nova tab');
    }
  }

  const handleEditTabTitle = async (id: number) => {
    if (!tabTitle) return toast.error('O título não pode ser vazio')
    if (id != selectedTab?.id) return toast.error('Erro ao editar título da tab')

    setEditMode(false)
    console.log(tabTitle)

    const data = {
      id: id,
      titulo: tabTitle,
    }

    try {
      const response = await api.put(`/tab`, data)
      console.log(response.data)
      setSelectedTab({ ...selectedTab, titulo: tabTitle })
      setTabs([...tabs.filter(tab => tab.id !== id), { ...selectedTab, titulo: tabTitle }])
    } catch {
      toast.error('Erro ao editar título da tab')
    }


  }

  const handleEditTab = (tab: Tab) => {

    setSelectedTab(tab)
    console.log('selected', tab)

    // Extract image element IDs
    const imageElementIds = tab.TabElements
      .filter(element => element.tipo === 'imagem')
      .map(imageElement => imageElement.id || 0); // Assuming the ID is a number

    setImageElementIds(imageElementIds);

    const simpleTextValue = tab.TabElements.find(element => element.tipo === 'texto')?.texto

    setSimpleText(simpleTextValue ?? '')

    const imgs = tab.TabElements.filter(item => item.tipo === 'imagem')
    const imgTitles = imgs.map(item => item.titulo)
    const imgVideo = tab.TabElements.find(element => element.tipo === 'video')

    setImgTitle(imgTitles.filter(title => title !== undefined).map(title => title ?? ''));
    setImgText(imgs.map(item => item.texto))
    setImgLink(imgs.filter(item => item.link != undefined).map(item => item?.link ?? ''))
    setVideoTitle(imgVideo?.titulo ?? '')
    setVideoText(imgVideo?.texto ?? '')
    setVideoLink(imgVideo?.link ?? '')
  };

  const handleRemoveTab = async (id: number | undefined) => {
    if (!id) return;
    try {
      const response = await api.delete(`/tab/${id}`);
      if (selectedTab?.id === id) {
        setSelectedTab(undefined)
      }
      console.log(response)
      toast.success('Tab removida com sucesso');
    } catch (error) {
      toast.error('Erro ao remover tab');
      console.error('Erro ao remover tab', error);
    }
    const newTabs = tabs.filter(tab => tab.id !== id);
    setTabs(newTabs);
  };

  // Upload img to api to get link
  const onImageChange = async (imageList: ImageListType) => {
    const newImageLinks: string[] = [];

    console.log(imageList)

    for (const image of imageList) {
      if (image) {
        const formData = new FormData();
        formData.append('file', image.file!);

        try {
          const response = await api.post('/archive', formData, {
            headers: {
              'Content-Type': 'multipart/form-data',
            },
          });

          const newLink = response.data;

          // Verifica se o link já existe no array
          if (!imgLink.includes(newLink)) {
            newImageLinks.push(newLink);
          } else {
            toast.warning('A imagem já foi adicionada anteriormente.');
          }
        } catch (error) {
          console.error('Erro ao processar resposta da API', error);
          toast.error('Erro ao processar resposta da API');
        }
      }
    }

    setImgLink([...imgLink, ...newImageLinks]);
  };

  const handleImageRemove = (index: number) => {
    const newImgLinks = imgLink.filter((item, i) => i !== index);
    setImgLink(newImgLinks);
  }

  const handleAddSimpleText = async (tab: Tab) => {
    const textElement = tab.TabElements.find(element => element.tipo === 'texto')

    if (!textElement?.id) {
      const data = {
        tab_id: selectedTab?.id!,
        tipo: 'texto',
        texto: simpleText
      }

      try {
        const response = await api.post('/tab-element', data)

        if (selectedTab) {
          setSelectedTab({
            ...selectedTab,
            TabElements: [...selectedTab.TabElements, response.data],
          });
        }
        toast.success('Texto adicionado com sucesso')

      } catch (error) {
        console.log(error)
        toast.error('Erro ao adicionar texto')
      }

    } else {

      const data = {
        id: textElement.id,
        tab_id: selectedTab?.id!,
        tipo: 'texto',
        texto: simpleText
      }

      try {
        const response = await api.put(`/tab-element`, data)
        console.log('put')
        console.log(response.data)
        if (selectedTab) {
          setSelectedTab({
            ...selectedTab,
            TabElements: [...selectedTab.TabElements.filter(element => element.id !== textElement.id), response.data],
          });
        }
        toast.success('Texto editado com sucesso')
      } catch (error) {
        console.log(error)
        toast.error('Erro ao editar texto')
      }

    }
  };

  const handleAddImage = async (tab: Tab, index: number) => {
    const imgElements = tab.TabElements.filter(element => element.tipo === 'imagem');

    if (imgElements[index]?.id) {
      setImageElementIds([...imageElementIds, imgElements[index].id as number]);
      const data = {
        id: imgElements[index].id,
        tab_id: selectedTab?.id!,
        tipo: 'imagem',
        titulo: imgTitle[index] || `Imagem ${index + 1}`,
        texto: imgText[index] || '',
        link: imgLink[index] || '',
      };

      try {
        const response = await api.put('/tab-element', data);

        if (selectedTab) {
          setSelectedTab({
            ...selectedTab,
            TabElements: [...selectedTab.TabElements.filter(element => element.id !== imgElements[index].id), response.data],
          });
        }
        toast.success('Imagem editada com sucesso');
      } catch (error) {
        console.log(error);
        toast.error('Erro ao editar imagem');
      }
    } else {

      const data = {
        tab_id: selectedTab?.id!,
        tipo: 'imagem',
        titulo: imgTitle[index] || `Imagem ${index + 1}`,
        texto: imgText[index] || '',
        link: imgLink[index] || '',
      };

      try {
        const response = await api.post('/tab-element', data);
        if (selectedTab) {
          setSelectedTab({
            ...selectedTab,
            TabElements: [...selectedTab.TabElements, response.data],
          });
        }
        toast.success('Imagem adicionada com sucesso');
      } catch (error) {
        console.log(error);
      }

    }
  };

  const handleRemoveTabImage = async (index: number) => {
    const imgElements = selectedTab?.TabElements.filter(element => element.tipo === 'imagem');
    const imgElement = imgElements?.find(element => element.id === imageElementIds[index]);

    if (imgElement?.id) {
      try {
        const response = await api.delete(`/tab-element/${imgElement.id}`);
        console.log(response);
        const newTabElements = selectedTab?.TabElements.filter(element => element.id !== imgElement.id);
        setSelectedTab({
          ...selectedTab!,
          TabElements: newTabElements || [],
        });
        const newImageElementIds = imageElementIds.filter((item, i) => i !== index);
        setImageElementIds(newImageElementIds);

        const newImgTitles = imgTitle.filter((item, i) => i !== index);
        setImgTitle(newImgTitles);

        const newImgTexts = imgText.filter((item, i) => i !== index);
        setImgText(newImgTexts);

        const newImgLinks = imgLink.filter((item, i) => i !== index);
        setImgLink(newImgLinks);

      } catch (error) {
        console.log(error);
      }
    }

    const newImageElementIds = imageElementIds.filter((item, i) => i !== index);
    setImageElementIds(newImageElementIds);

  }

  const handleAddVideo = async (tab: Tab) => {
    const videoElement = tab.TabElements.find(element => element.tipo === 'video')

    if (!videoElement?.id) {
      const data = {
        tab_id: selectedTab?.id!,
        tipo: 'video',
        titulo: videoTitle,
        texto: videoText,
        link: videoLink,
      }

      try {
        const response = await api.post('/tab-element', data)

        if (selectedTab) {
          setSelectedTab({
            ...selectedTab,
            TabElements: [...selectedTab.TabElements, response.data],
          });
        }
        toast.success('Video adicionado com sucesso')
      } catch (error) {
        console.log(error)
        toast.error('Erro ao adicionar video')
      }

    } else {

      const data = {
        id: videoElement.id,
        tab_id: selectedTab?.id!,
        tipo: 'video',
        titulo: videoTitle,
        texto: videoText,
        link: videoLink,
      }

      try {
        const response = await api.put(`/tab-element`, data)

        if (selectedTab) {
          setSelectedTab({
            ...selectedTab,
            TabElements: [...selectedTab.TabElements.filter(element => element.id !== videoElement.id), response.data],
          });
        }
        toast.success('Video editado com sucesso')
      } catch (error) {
        console.log(error)
        toast.error('Erro ao editar video')
      }
    }

  }

  const handleRemoveVideo = async () => {
    const videoElement = selectedTab?.TabElements.find(element => element.tipo === 'video')

    if (videoElement?.id) {
      try {
        const response = await api.delete(`/tab-element/${videoElement.id}`);
        console.log(response);
        const newTabElements = selectedTab?.TabElements.filter(element => element.id !== videoElement.id);
        setSelectedTab({
          ...selectedTab!,
          TabElements: newTabElements || [],
        });

        setVideoTitle('')
        setVideoLink('')
        setVideoText('')

      } catch (error) {
        console.log(error);
      }
    }
  }


  const handleImgTitleChange = (e: React.ChangeEvent<HTMLInputElement>, index: number) => {
    const newImgTitles = [...imgTitle];
    newImgTitles[index] = e.target.value;
    setImgTitle(newImgTitles);
    console.log('newImgTitles', newImgTitles)
  };

  const handleImgTextChange = (e: React.ChangeEvent<HTMLTextAreaElement>, index: number) => {
    const newImgTexts = [...imgText];
    newImgTexts[index] = e.target.value;
    setImgText(newImgTexts);
    console.log('newImgTexts', newImgTexts)
  };

  useEffect(() => {
    console.log("tabs", tabs)
  }, [tabs])

  useEffect(() => {
    console.log("selectedTab", selectedTab)
  }, [selectedTab])

  return (
    <div className='w-100 position-relative bg-light overflow-hidden rounded-4 p-3' style={{ height: "600px" }}>
      <Row>
        <h4 className='text-center mb-4'>Descrição do produto</h4>
        <Col lg={5}>
          <div>
            <ProductCard id={id} name={name} image={image} />
            <div className='card'>
              <div className='card-body'>
                <h5 className='card-title'>Lista de tabs</h5>
                {tabs?.map((tab, index) => (
                  <div key={index} className='d-flex align-items-center justify-content-between mb-2'>
                    <p className='card-text mb-0'>{tab?.titulo}</p>
                    <div>
                      {tab.id === selectedTab?.id ? (
                        <></>
                      ) : (
                        <button onClick={e => handleEditTab(tab)} className='btn btn-primary me-2'>Editar</button>

                      )}
                      <button onClick={e => handleRemoveTab(tab?.id)} className='btn btn-danger'>Remover</button>
                    </div>
                  </div>
                ))}
                <button onClick={e => handleAddTab()} className='w-100 btn btn-primary'>Criar</button>
              </div>
            </div>
          </div>


        </Col>

        <Col lg={7}>
          {selectedTab && (
            <div className='card position-relative overflow-scroll p-3' style={{ height: '520px' }}>
              {editMode ? (
                <div className='d-flex align-items-center justify-content-center mb-3'>
                  <input onChange={e => setTabTitle(e.target.value)} className='w-50 form-control me-2' type="text" placeholder={selectedTab.titulo} />
                  <button onClick={e => handleEditTabTitle(selectedTab.id!)} className='btn btn-success'>
                    <Image src='/svg/save.svg' width={24} height={24} alt='' />
                  </button>
                </div>
              ) :
                (
                  <div className='d-flex align-items-center justify-content-center mb-3'>
                    <h4 className='card-title text-center me-2 mb-0'>{selectedTab.titulo}</h4>
                    <button onClick={e => setEditMode(true)} className='d-flex btn btn-primary p-2'>
                      <Image src='/svg/edit.svg' width={16} height={16} alt='' />
                    </button>
                  </div>
                )}
              <Tabs
                defaultActiveKey="simples"
                className="mb-3"
              >
                {/* Simples */}
                <Tab eventKey="simples" title="Texto simples">
                  <div className='form-floating'>
                    <textarea onChange={e => setSimpleText(e.target.value)} value={simpleText} className='form-control' name="" id="" rows={10}>
                    </textarea>
                    <label className='mb-2'>Conteúdo do texto</label>
                  </div>
                  <div className='d-flex justify-content-end'>

                    <button onClick={e => handleAddSimpleText(selectedTab)} className='btn btn-primary mt-3'>Salvar</button>
                  </div>
                </Tab>
                {/* Imagem 1 */}
                <Tab eventKey={`imagem_1`} title={`Imagem 1`}>
                  <div className="mb-3">
                    <ImageUploading
                      multiple
                      value={images}
                      onChange={onImageChange}
                      maxNumber={1}
                    >
                      {({
                        onImageUpdate,
                        isDragging,
                        dragProps,
                      }) => (
                        <div className="w-100">
                          <div className="rounded-4">
                            <div className="w-100 d-flex flex-column align-items-center mb-3">
                              <h6>Imagem 1</h6>
                              <div
                                className="position-relative bg-secondary bg-opacity-25 rounded-4 mb-3"
                                style={{ width: '280px', height: '280px' }}
                              >
                                {!imgLink[0] && (
                                  <div
                                    className="w-100 h-100 d-flex align-items-center text-center"
                                    onClick={() => onImageUpdate(0)}
                                    style={isDragging ? { color: 'red' } : undefined}
                                    {...dragProps}
                                  >
                                    Clique aqui ou arraste e solte uma imagem
                                  </div>
                                )}
                                {imgLink[0] ? (
                                  <img
                                    src={imgLink[0]}
                                    className="w-100 h-100 object-fit-contain"
                                    alt=""
                                  />
                                ) : null}
                                <button
                                  onClick={() => handleImageRemove(0)}
                                  className="position-absolute top-1 end-1 btn btn-danger"
                                >
                                  <Image src="/svg/trash.svg" width={24} height={24} alt="" />
                                </button>
                              </div>
                              <div className="w-100 form-floating mb-3">
                                <input
                                  onChange={(e) => handleImgTitleChange(e, 0)}
                                  value={imgTitle[0] || ''}
                                  className="form-control"
                                  type="text"
                                />
                                <label>Título</label>
                              </div>
                              <div className="w-100 form-floating">
                                <textarea
                                  onChange={(e) => handleImgTextChange(e, 0)}
                                  value={imgText[0] || ''}
                                  className="form-control"
                                />
                                <label>Conteúdo do Texto</label>
                              </div>
                              <div className='mt-3'>

                                <button
                                  onClick={() => handleAddImage(selectedTab, 0)}
                                  className="btn btn-primary me-2"
                                >
                                  Salvar
                                </button>
                                <button onClick={e => handleRemoveTabImage(0)} className='btn btn-danger'>Excluir aba</button>
                              </div>
                            </div>
                          </div>
                        </div>
                      )}
                    </ImageUploading>
                  </div>
                </Tab>
                {/* Imagem 2 */}
                <Tab eventKey={`imagem_2`} title={`Imagem 2`}>
                  <div className="mb-3">
                    <ImageUploading
                      multiple
                      value={images}
                      onChange={onImageChange}
                      maxNumber={1}
                    >
                      {({
                        onImageUpdate,
                        isDragging,
                        dragProps,
                      }) => (
                        <div className="w-100">
                          <div className="rounded-4">
                            <div className="w-100 d-flex flex-column align-items-center mb-3">
                              <h6>Imagem 1</h6>
                              <div
                                className="position-relative bg-secondary bg-opacity-25 rounded-4 mb-3"
                                style={{ width: '280px', height: '280px' }}
                              >
                                {!imgLink[1] && (
                                  <div
                                    className="w-100 h-100 d-flex align-items-center text-center"
                                    onClick={() => onImageUpdate(0)}
                                    style={isDragging ? { color: 'red' } : undefined}
                                    {...dragProps}
                                  >
                                    Clique aqui ou arraste e solte uma imagem
                                  </div>
                                )}
                                {imgLink[1] ? (
                                  <img
                                    src={imgLink[1]}
                                    className="w-100 h-100 object-fit-contain"
                                    alt=""
                                  />
                                ) : null}
                                <button
                                  onClick={() => handleImageRemove(1)}
                                  className="position-absolute top-1 end-1 btn btn-danger"
                                >
                                  <Image src="/svg/trash.svg" width={24} height={24} alt="" />
                                </button>
                              </div>
                              <div className="w-100 form-floating mb-3">
                                <input
                                  onChange={(e) => handleImgTitleChange(e, 1)}
                                  value={imgTitle[1] || ''}
                                  className="form-control"
                                  type="text"
                                />
                                <label>Título</label>
                              </div>
                              <div className="w-100 form-floating">
                                <textarea
                                  onChange={(e) => handleImgTextChange(e, 1)}
                                  value={imgText[1] || ''}
                                  className="form-control"
                                />
                                <label>Conteúdo do Texto</label>
                              </div>
                              <div className='mt-3'>
                                <button
                                  onClick={() => handleAddImage(selectedTab, 1)}
                                  className="btn btn-primary me-2"
                                >
                                  Salvar
                                </button>
                                <button onClick={e => handleRemoveTabImage(1)} className='btn btn-danger'>Excluir aba</button>
                              </div>
                            </div>
                          </div>
                        </div>
                      )}
                    </ImageUploading>
                  </div>
                </Tab>
                {/* Imagem 3 */}
                <Tab eventKey={`imagem_3`} title={`Imagem 3`}>
                  <div className="mb-3">
                    <ImageUploading
                      multiple
                      value={images}
                      onChange={onImageChange}
                      maxNumber={1}
                    >
                      {({
                        imageList,
                        onImageUpload,
                        onImageRemove,
                        onImageUpdate,
                        isDragging,
                        dragProps,
                      }) => (
                        <div className="w-100">
                          <div className="rounded-4">
                            <div className="w-100 d-flex flex-column align-items-center mb-3">
                              <h6>Imagem 1</h6>
                              <div
                                className="position-relative bg-secondary bg-opacity-25 rounded-4 mb-3"
                                style={{ width: '280px', height: '280px' }}
                              >
                                {!imgLink[2] && (
                                  <div
                                    className="w-100 h-100 d-flex align-items-center text-center"
                                    onClick={() => onImageUpdate(2)}
                                    style={isDragging ? { color: 'red' } : undefined}
                                    {...dragProps}
                                  >
                                    Clique aqui ou arraste e solte uma imagem
                                  </div>
                                )}
                                {imgLink[2] ? (
                                  <img
                                    src={imgLink[2]}
                                    className="w-100 h-100 object-fit-contain"
                                    alt=""
                                  />
                                ) : null}
                                <button
                                  onClick={() => handleImageRemove(2)}
                                  className="position-absolute top-1 end-1 btn btn-danger"
                                >
                                  <Image src="/svg/trash.svg" width={24} height={24} alt="" />
                                </button>
                              </div>
                              <div className="w-100 form-floating mb-3">
                                <input
                                  onChange={(e) => handleImgTitleChange(e, 2)}
                                  value={imgTitle[2] || ''}
                                  className="form-control"
                                  type="text"
                                />
                                <label>Título</label>
                              </div>
                              <div className="w-100 form-floating">
                                <textarea
                                  onChange={(e) => handleImgTextChange(e, 2)}
                                  value={imgText[2] || ''}
                                  className="form-control"
                                />
                                <label>Conteúdo do Texto</label>
                              </div>
                              <div className='mt-3'>
                                <button
                                  onClick={() => handleAddImage(selectedTab, 2)}
                                  className="btn btn-primary me-2"
                                >
                                  Salvar
                                </button>
                                <button onClick={e => handleRemoveTabImage(2)} className='btn btn-danger'>Excluir aba</button>
                              </div>
                            </div>
                          </div>
                        </div>
                      )}
                    </ImageUploading>
                  </div>
                </Tab>

                {/* Video */}
                <Tab eventKey="video" title="Video">
                  <div className='form-floating mb-3'>
                    <input onChange={e => setVideoTitle(e.target.value)} value={videoTitle} className='form-control' type="text" name="" id="" />
                    <label htmlFor="">Titulo</label>
                  </div>
                  <div className='form-floating mb-3'>
                    <input onChange={e => setVideoLink(e.target.value)} value={videoLink} className='form-control' type="text" name="" id="" />
                    <label htmlFor="">Link</label>
                  </div>
                  <div className='form-floating mb-3'>
                    <input onChange={e => setVideoText(e.target.value)} value={videoText} className='form-control' type="text" name="" id="" />
                    <label htmlFor="">Texto</label>
                  </div>
                  <div className='d-flex justify-content-end'>

                    <button onClick={e => handleAddVideo(selectedTab)} className='btn btn-primary me-2'>Salvar</button>
                    <button onClick={e => handleRemoveVideo()} className='btn btn-danger'>Excluir</button>
                  </div>
                </Tab>
              </Tabs>
            </div>
          )}
        </Col>

      </Row>
      <ToastContainer />
      <button onClick={e => window.history.back()} className='btn btn-dark position-absolute bottom-0 mb-3'>Voltar</button>
    </div>
  )
}

export default NewDescriptionProduct
