import React, { use, useEffect, useState } from 'react'
import { Row, Col } from 'react-bootstrap'
import ProductCard from './../ProductCard/ProductCard';

import { HexColorPicker } from "react-colorful";
import Image from 'next/image';

import { toast, ToastContainer } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

import ReactCrop, { type Crop } from 'react-image-crop'
import 'react-image-crop/dist/ReactCrop.css'

import ImageUploading, { ImageListType } from 'react-images-uploading';
import api from '@/utils/api';
import VariantChildList from './../VariantChildList/VariantChildList';

interface Props {
    id: number;
    name: string;
    image: string;
}

interface Item {
    productId: number;
    nome: string;
    tipo: string;
    fonte: boolean;
    cor: boolean;
    imagem: boolean;
    tamanhoFonte?: number;
    ocultoCart: boolean;
    required: boolean;
    preview: boolean;
    width?: number;
    height?: number;
    x?: number;
    y?: number;
    varValues: Values[];
}
interface Values {
    valor: string;
    referencia: string;
    placeholder?: string;
}

interface ChildValues {
    variant_filha_id: number
    valor: string
    referencia: string
}


const PreDeterminated: React.FC<Props> = ({ id, name, image }) => {

    // inputs
    const [customName, setCustomName] = useState<string>('')
    const [variantName, setVariantName] = useState<string>('')
    const [variantPlaceholder, setVariantPlaceholder] = useState<string>('')
    const [variantValue, setVariantValue] = useState<string>('')
    const [fontSize, setFontSize] = useState<number>()

    const [type, setType] = useState('Cor')

    const [color, setColor] = useState("#000000");
    const [uploadedImage, setUploadedImage] = useState<string | null>(null);

    const [preview, setPreview] = useState(false)
    const [hide, setHide] = useState(false)
    const [required, setRequired] = useState(false)
    const [variantChild, setVariantChild] = useState<boolean>(false)

    const [valuesList, setValuesList] = useState<Values[]>([])
    const [variantChildValues, setVariantChildValues] = useState<ChildValues[]>([])

    // Crop image
    const [crop, setCrop] = useState<Crop>({ x: 0, y: 0, width: 100, height: 100, unit: '%' })
    const [images, setImages] = useState([])



    const handleImageUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {

        const file = e.target.files?.[0];

        if (file) {
            const imageUrl = URL.createObjectURL(file);
            setUploadedImage(imageUrl);

        }
        try {
            const formData = new FormData();
            formData.append('file', file!);

            // Upload the image using FormData
            const response = await api.post('/archive', formData, {
                headers: {
                    'Content-Type': 'multipart/form-data',
                },
            });
            setVariantValue(response.data)
            console.log(response.data)
        } catch (error) {
            console.log(error)
        }

    };

    const addValue = () => {
        if (customName === '') return toast.error('Nome da variante é obrigatório')
        if (variantName === '') return toast.error('Nome da variação é obrigatório')
        if (variantValue === '') return toast.error('Referência da variação é obrigatório')
        if ((type === 'Texto simples' || type === 'Imagem') && preview && !crop) return toast.error('Selecione uma imagem para a pré visualização')

        let value: Values = {
            valor: variantName,
            referencia: variantValue,
            placeholder: variantPlaceholder
        }

        if (type === 'Texto simples' || variantPlaceholder === '') {
            value = {
                valor: variantName,
                referencia: variantValue,
            }
        }

        setValuesList([...valuesList, value])
        cleanInputs()
    }

    useEffect(() => {
        console.log('valuesList', valuesList)
    }, [valuesList])

    const removeValue = (index: number) => {
        const newList = valuesList.filter((value, i) => i !== index)
        setValuesList(newList)
    }

    const cleanInputs = () => {
        setVariantName('')
        setVariantPlaceholder('')
        setVariantValue('')
        setUploadedImage(null)
    }

    const cleanValues = () => {
        setValuesList([])
        setCrop({ x: 0, y: 0, width: 100, height: 100, unit: '%' })
        setImages([])
    }


    const sendValues = async () => {
        if (valuesList.length === 0 && variantChildValues.length === 0) return toast.error('Adicione pelo menos uma variação')
        if (customName === '') return toast.error('Nome da variante é obrigatório')
        let data: Item = {
            productId: id,
            nome: customName,
            tipo: 'predeterminada',
            fonte: type === 'Fonte',
            cor: type === 'Cor',
            imagem: type === 'Imagem',
            tamanhoFonte: fontSize,
            ocultoCart: variantChild ? true : hide,
            required: required,
            preview: preview,
            varValues: !variantChild ? valuesList : variantChildValues,
        }

        if ((type === 'Simples' || type === 'Imagem') && preview) {
            data = {
                ...data,
                width: crop?.width,
                height: crop?.height,
                x: crop?.x,
                y: crop?.y,
            }
        }
        console.log('sendData', data)
        try {
            const response = await api.post('/variant', data)
            toast.success('Variação criada com sucesso')
            console.log(response)
            setValuesList([])
            setVariantChildValues([])
            setUploadedImage(null)
            setImages(images as never[])
        } catch (error) {
            toast.error('Erro ao criar variação')
            console.log(error)
        }
        console.log(data)
    }


    const onImageChange = (
        imageList: ImageListType,
        addUpdateIndex: number[] | undefined
    ) => {
        // data for submit
        setImages(imageList as never[])
    };

    const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const { name, value } = e.target;
        setCrop(prevCrop => ({ ...prevCrop, [name]: parseInt(value, 10) }));
    };

    return (
        <div className='w-80 bg-light rounded-4 p-3'>
            <Row>
                <Col lg={6}>
                    <div>
                        <h4>Pré determinada</h4>
                        <div className='mb-2'>
                            <label htmlFor="">Nome:</label>
                            <input onChange={e => setCustomName(e.target.value)} value={customName} className='form-control' type="text" />
                        </div>
                        {!variantChild && (
                            <div className='mb-2'>
                                <label htmlFor="">Nome da variação:</label>
                                <input onChange={e => setVariantName(e.target.value)} value={variantName} className='form-control' type="text" />
                            </div>
                        )}
                        {type === 'Simples' && !variantChild && (
                            <div className='mb-2'>
                                <label htmlFor="">Referencia:</label>
                                <input onChange={e => setVariantValue(e.target.value)} value={variantValue} className='form-control' type="text" />
                            </div>
                        )}
                        {type === 'Simples' && preview && (
                            <div className='mb-2'>
                                <label htmlFor="">Tamanho da fonte (px):</label>
                                <input onChange={e => setFontSize(Number(e.target.value))} className='form-control' type="number" />
                            </div>

                        )}
                        {type != 'Simples' && (
                            <div className='mb-2'>
                                <label htmlFor="">Placeholder (opcional):</label>
                                <input onChange={e => setVariantPlaceholder(e.target.value)} value={variantPlaceholder} className='form-control' type="text" />
                            </div>
                        )}
                        {/* Image input */}
                        {type === 'Imagem' && !preview && (
                            <div className='mb-2'>
                                <label className='mb-1'>Imagem:</label>
                                <input onChange={handleImageUpload} className='form-control' type="file" />
                            </div>
                        )}
                        {type === 'Imagem' && preview && (
                            <div className='mb-2'>
                                <label className='mb-1'>Código SVG:</label>
                                <input onChange={e => setVariantValue(e.target.value)} value={variantValue} className='form-control' type="text" />
                            </div>
                        )}
                        {/* Color input */}
                        {type === 'Fonte' && (
                            <div className='mb-2'>
                                <label htmlFor="">Link da fonte:</label>
                                <input onChange={e => setVariantValue(e.target.value)} value={variantValue} className='form-control' type="text" placeholder='' />
                            </div>
                        )}
                        <div className='d-flex mb-3'>
                            {/* Type input */}
                            <div className='me-4'>
                                <label className='mb-1'>Tipo:</label>
                                <div className='form-check'>
                                    <label htmlFor="">Cor</label>
                                    <input onChange={e => { setType(e.target.value), cleanValues() }} className='form-check-input' value="Cor" type="radio" name='type' disabled={valuesList.length > 0 || variantChild} defaultChecked />
                                </div>
                                <div className='form-check'>
                                    <label htmlFor="">Imagem</label>
                                    <input onChange={e => { setType(e.target.value), cleanValues() }} className='form-check-input' value="Imagem" type="radio" name='type' disabled={valuesList.length > 0 || variantChild} />
                                </div>
                                <div className='form-check'>
                                    <label htmlFor="">Fonte</label>
                                    <input onChange={e => { setType(e.target.value), cleanValues() }} className='form-check-input' value="Fonte" type="radio" name='type' disabled={valuesList.length > 0 || variantChild} />
                                </div>
                                <div className='form-check'>
                                    <label htmlFor="">Texto simples</label>
                                    <input onChange={e => { setType(e.target.value), cleanValues() }} className='form-check-input' value="Simples" type="radio" name='type' disabled={valuesList.length > 0} />
                                </div>
                            </div>
                            {/* Options */}
                            <div>
                                <label className='mb-1'>Opções:</label>
                                <div className='form-check'>
                                    <label htmlFor="">Pré visualização</label>
                                    <input onChange={e => { setPreview(!preview), cleanInputs() }} checked={preview} className='form-check-input' type="checkbox" name='' disabled={valuesList.length > 0 || variantChild} />
                                </div>
                                <div className='form-check'>
                                    <label htmlFor="">Oculto no carrinho</label>
                                    <input onChange={e => setHide(!hide)} className='form-check-input' type="checkbox" name='' disabled={valuesList.length > 0 || variantChild} />
                                </div>
                                <div className='form-check'>
                                    <label htmlFor="">Obrigatório</label>
                                    <input onChange={e => setRequired(!required)} className='form-check-input' type="checkbox" name='' disabled={valuesList.length > 0 || variantChild} />
                                </div>
                                <div className='form-check'>
                                    <label htmlFor="">Variação pai</label>
                                    <input onChange={e => { setVariantChild(!variantChild), setValuesList([]), setVariantChildValues([]) }} className='form-check-input' type="checkbox" name='' disabled={valuesList.length > 0 || type !== 'Simples'} />
                                </div>
                            </div>
                        </div>

                        <div>
                            {!variantChild && (
                                <button onClick={e => addValue()} className='btn btn-primary fs-8 mb-3'>Adicionar +</button>
                            )}
                            {valuesList.length > 0 && (
                                <div className='overflow-y-scroll fs-7' style={{ maxHeight: '200px' }}>
                                    <h5>Valores adicionados:</h5>
                                    {valuesList.map((item, index) => (
                                        <div className='d-flex align-items-center mb-2' key={index}>
                                            <div className='d-flex'>
                                                <div className='bg-secondary bg-opacity-25 text-ellipsis rounded fs-8 px-2 py-1 me-2'><b>Nome:</b> {item.valor}</div>
                                                <div className='d-flex align-items-center bg-secondary bg-opacity-25 text-ellipsis rounded fs-8 px-2 py-1 me-2'><b>{type}:</b>
                                                    {type != 'Imagem' && (
                                                        <div className='text-ellipsis'>
                                                            {item.referencia}
                                                        </div>
                                                    )}
                                                    {type === 'Imagem' && !preview && (
                                                        <Image src={item.referencia} width={36} height={36} alt='' />
                                                    )}
                                                    {type === 'Imagem' && preview && (
                                                        <>
                                                            {item.referencia && (
                                                                <div className='text-ellipsis'>
                                                                    {item.referencia}
                                                                </div>
                                                            )}
                                                        </>
                                                    )}
                                                    {type === 'Cor' && (
                                                        <div className='text-ellipsis rounded border p-1 ms-2' style={{ width: '20px', height: '20px', background: `${item.referencia}` }}></div>
                                                    )}
                                                </div>
                                                {item.placeholder && (
                                                    <div className='bg-secondary bg-opacity-25 text-ellipsis rounded fs-8 px-2 py-1 me-2'><b>Placeholder:</b> {item.placeholder}</div>
                                                )}
                                            </div>

                                            <button onClick={e => removeValue(index)} className='d-flex align-items-center btn btn-danger'>
                                                <Image src='/svg/trash.svg' width={16} height={16} alt='' />
                                            </button>
                                        </div>
                                    ))}
                                </div>
                            )}

                            {variantChildValues.length > 0 && (
                                <>
                                    {variantChildValues.map((child, index) => (
                                        <div key={index} className='d-flex mb-2'>
                                            <div className='bg-secondary bg-opacity-25 text-ellipsis rounded fs-8 px-2 py-1 me-2'><b>ID: </b>{child.variant_filha_id}</div>
                                            <div className='bg-secondary bg-opacity-25 text-ellipsis rounded fs-8 px-2 py-1 me-2'><b>Valor: </b>{child.valor}</div>
                                            <div className='bg-secondary bg-opacity-25 text-ellipsis rounded fs-8 px-2 py-1 me-2'><b>Refernência: </b>{child.referencia}</div>
                                        </div>
                                    ))}
                                </>
                            )}
                        </div>
                    </div>
                </Col>
                <Col lg={6}>
                    <div className='overflow-y-scroll' style={{ maxHeight: '500px' }}>
                        <ProductCard id={id} name={name} image={image} />
                        {/* Color picker */}
                        {(type === 'Cor') && (
                            <>
                                <div className='d-flex'>
                                    <div className='d-flex align-items-center fs-7 m-3'>
                                        <label className='me-2'>HEX:</label>
                                        <input onChange={e => { setColor(e.target.value), setVariantValue(e.target.value) }} value={variantValue} type="text" className='form-control me-2' placeholder='ex: #FF0000 ou Red' />
                                        <div className='rounded border me-1' style={{ width: '36px', height: '36px', backgroundColor: `${color}` }}></div>
                                    </div>
                                </div>
                            </>
                        )}
                        {/* Image preview */}
                        {type === "Imagem" && uploadedImage && (
                            <div className='mt-3'>
                                <h6>Preview da Imagem</h6>
                                <img src={uploadedImage ?? ''} alt="Preview" className="img-fluid" width={90} height={90} />
                            </div>
                        )}

                        {((type === 'Simples' && !variantChild) || type === 'Imagem') && preview && (
                            <div className='d-flex flex-column align-items-center'>
                                <div className='d-flex my-4'>
                                    <div className='d-flex flex-column me-2'>
                                        <div className='form-group'>
                                            <label className='fs-8'>Largura:</label>
                                            <input onChange={e => handleInputChange(e)} value={crop.width} name='width' type='number' className='form-control bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2 mb-2' />
                                        </div>
                                        <div>
                                            <label className='fs-8'>Posição X %:</label>
                                            <input onChange={e => handleInputChange(e)} value={crop.x} name='x' type='number' className='form-control bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2 mb-2' />
                                        </div>
                                    </div>
                                    <div className='d-flex'>
                                        <div className='d-flex flex-column me-5'>
                                            <div className='form-group'>
                                                <label className='fs-8'>Altura:</label>
                                                <input onChange={e => handleInputChange(e)} value={crop.height} name='height' type='number' className='form-control bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2 mb-2' />
                                            </div>
                                            <div className='form-group'>
                                                <label className='fs-8'>Posição Y %:</label>
                                                <input onChange={e => handleInputChange(e)} value={crop.y} name='y' type='number' className='form-control bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2' />
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <ImageUploading
                                    multiple
                                    value={images}
                                    onChange={onImageChange}
                                    maxNumber={1}
                                >
                                    {({
                                        imageList,
                                        onImageUpload,
                                        onImageRemove,
                                        isDragging,
                                        dragProps,
                                    }) => (
                                        // write your building UI
                                        <div className="w-100 d-flex align-items-center justify-content-center px-2">
                                            <div className='d-flex align-items-center position-relative bg-secondary bg-opacity-25 rounded-4' style={{ minWidth: '200px', minHeight: '100px' }}>
                                                {!imageList[0] && (
                                                    <div
                                                        className='text-center fs-8 p-3'
                                                        onClick={onImageUpload}
                                                        style={isDragging ? { color: 'red' } : undefined}
                                                        {...dragProps}
                                                    >
                                                        Clique aqui ou arraste e solte uma imagem
                                                    </div>
                                                )}
                                                {imageList[0] && (
                                                    <ReactCrop crop={crop} onChange={(c, p) => setCrop(p)}>
                                                        <img className='position-relative rounded-4' src={imageList[0].dataURL} style={{ objectFit: 'contain' }} alt='' />
                                                    </ReactCrop>
                                                )}
                                                <button className='btn btn-danger position-absolute top-0 right-0 m-2 px-1 py-0' onClick={() => onImageRemove(0)}>
                                                    <Image src='/svg/trash.svg' width={16} height={16} alt='' />
                                                </button>
                                            </div>
                                        </div>
                                    )}
                                </ImageUploading>
                            </div>
                        )}

                        {variantChild && (
                            <VariantChildList productId={id} variantChildValues={variantChildValues} setVariantChildValues={setVariantChildValues} />
                        )}

                        <div className='d-flex justify-content-end mt-3'>
                            <button onClick={e => sendValues()} className='d-flex align-items-center btn btn-success me-2'>Salvar
                                <Image src='/svg/save.svg' width={24} height={24} alt='' />
                            </button>
                            <button onClick={e => { window.history.back() }} className='btn btn-dark'>
                                Voltar
                            </button>
                        </div>
                    </div>
                </Col>
            </Row>
            <ToastContainer />
        </div>
    )
}

export default PreDeterminated