import React, { useEffect, useState } from 'react';
import Image from 'next/image';
import { Row, Col } from 'react-bootstrap';
import ProductCard from '../ProductCard/ProductCard';

import ReactCrop, { type Crop } from 'react-image-crop'
import 'react-image-crop/dist/ReactCrop.css'

import ImageUploading, { ImageListType } from 'react-images-uploading';

import { toast, ToastContainer } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';
import api from '@/utils/api';

interface Props {
    id: number;
    name: string;
    image: string;
}

interface Values {
    tipo: string;
    nome: string;
    limite?: number;
    fonte: boolean;
    imagem: boolean;
    tamanhoFonte?: number;
    cor: boolean;
    ocultoCart: boolean;
    preview: boolean;
    required: boolean;
    x?: number | null;
    y?: number | null;
    width?: number | null;
    height?: number | null;
    productId: number;
}

const SimpleText: React.FC<Props> = ({ id, name, image }) => {
    // Upload image
    const maxNumber = 69;
    const [images, setImages] = useState([]);

    // Checkboxes
    const [preview, setPreview] = useState<boolean>(false)
    const [hideCart, setHideCart] = useState<boolean>(false)
    const [required, setRequired] = useState<boolean>(false)

    // Crop image
    const [crop, setCrop] = useState<Crop>({ x: 0, y: 0, width: 100, height: 100, unit: '%' });

    const [nameValue, setNameValue] = useState<string>('')
    const [limit, setLimit] = useState<number>()
    const [fontSize, setFontSize] = useState<number>()
    const [valuesList, setValuesList] = useState<Values>()

    const onImageChange = (
        imageList: ImageListType,
        addUpdateIndex: number[] | undefined
    ) => {
        // data for submit
        setImages(imageList as never[]);
    };

    const addNewValue = () => {
        if (!nameValue) return toast.error('Preencha o nome do texto')
        if (preview && !crop) return toast.error('Selecione uma imagem e recorte')

        let values: Values = {
            tipo: "simples",
            nome: nameValue,
            limite: limit,
            fonte: false,
            tamanhoFonte: fontSize,
            imagem: false,
            cor: false,
            ocultoCart: hideCart,
            preview: preview,
            required: required,
            productId: id,
        };

        if (preview && crop) {
            values = {
                ...values,
                x: crop.x,
                y: crop.y,
                width: crop.width,
                height: crop.height,
            };
        }
        setValuesList(values);
        setImages([]);
        setCrop({ x: 0, y: 0, width: 0, height: 0, unit: '%' });
    }

    useEffect(() => {
        console.log('valuesList', valuesList)
    }, [valuesList])

    const cleanValues = () => {
        setNameValue('')
        setLimit(undefined)
        setImages([])
        setCrop({ x: 0, y: 0, width: 0, height: 0, unit: '%' });
    }

    const sendVariants = async () => {
        const data = valuesList;
        try {
            const response = await Promise.resolve(api.post('/variant', data))
            console.log(response)
            toast.success('Variações salvas com sucesso')
            cleanValues()
            setValuesList(undefined)
        } catch (error) {
            console.log(error)
            toast.error('Erro ao salvar variações')
        }

    }

    useEffect(() => {
        console.log('valuesList', valuesList)
    }, [valuesList])

    const handleBack = () => {
        window.history.back();
    }

    const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const { name, value } = e.target;
        setCrop(prevCrop => ({ ...prevCrop, [name]: parseInt(value, 10) }));
    };


    return (
        <div className="w-80">
            <div className='bg-light rounded-4 p-3'>
                <Row>
                    <Col lg={7}>
                        <div>
                            <h4 className='fw-bold mb-4 mb-4'>Texto simples</h4>
                            <div>
                                <div className=''>
                                    <label className="mb-2" htmlFor="">Nome do texto</label>
                                    <input value={nameValue} onChange={e => setNameValue(e.target.value)} className='form-control' type="text" />
                                </div>
                                <div className='w-50 mb-3'>
                                    <label className="mb-2" htmlFor="">Limite de caracteres</label>
                                    <input value={limit === undefined ? '' : limit} onChange={e => setLimit(Number(e.target.value))} className='form-control' type="number" />
                                </div>
                                {preview && (
                                    <div className='w-50 mb-3'>
                                        <label className="mb-2" htmlFor="">Tamanho da fonte (px): </label>
                                        <input onChange={e => setFontSize(Number(e.target.value))} className='form-control' type="number" />
                                    </div>
                                )}
                                <div className='mb-3 border-bottom pb-2'>
                                    <div className='form-check'>
                                        <label htmlFor="">Pré-visualização</label>
                                        <input className='form-check-input' onChange={(e) => setPreview(!preview)} checked={preview} type="checkbox" name="" id="previewCheck" />
                                    </div>
                                    <div className='form-check'>
                                        <label htmlFor="">Ocultar no carrinho</label>
                                        <input className='form-check-input' onChange={(e) => setHideCart(!hideCart)} type="checkbox" name="" id="hideCartCheck" />
                                    </div>
                                    <div className='form-check'>
                                        <label htmlFor="">Obrigatório</label>
                                        <input className='form-check-input' onChange={(e) => setRequired(!required)} type="checkbox" name="" id="requiredCheck" />
                                    </div>
                                    <div className='my-3'>
                                        <button className='btn btn-primary fs-8' onClick={e => addNewValue()} >Adicionar +</button>
                                    </div>
                                </div>
                                {preview && (
                                    <div className='d-flex align-items-center justify-content-between border-bottom mb-3'>
                                        {/* Dimensions */}
                                        <div className='d-flex mb-3'>
                                            <div className='me-2'>
                                                <div className='mb-2'>
                                                    <span className='fw-bold me-2'>X:</span>
                                                    <input onChange={e => handleInputChange(e)} value={crop.x} name='x' className='w-50 bg-secondary bg-opacity-25 border-0 rounded fs-8 px-3 py-1' type='number' />
                                                </div>
                                                <div>
                                                    <span className='fw-bold me-2'>Y:</span>
                                                    <input onChange={e => handleInputChange(e)} value={crop.y} name='y' className='w-50 bg-secondary bg-opacity-25 border-0  rounded fs-8 px-3 py-1' type='number' />
                                                </div>
                                            </div>
                                            <div>
                                                <div className='mb-2'>
                                                    <span className='fw-bold me-2'>Largura:</span>
                                                    <input onChange={e => handleInputChange(e)} value={crop.width} name='width' className='w-50 bg-secondary bg-opacity-25 border-0  rounded fs-8 px-3 py-1' type='number' />
                                                </div>
                                                <div>
                                                    <span className='fw-bold me-2'>Altura:</span>
                                                    <input onChange={e => handleInputChange(e)} value={crop.height} name='height' className='w-50 bg-secondary bg-opacity-25 border-0  rounded fs-8 px-3 py-1' type='number' />
                                                </div>
                                            </div>
                                        </div>


                                    </div>
                                )}


                                <div className='d-flex align-items-center bg-opacity-25 rounded-4'>
                                    {valuesList && (
                                        <div className='w-100'>
                                            <h6>Valores adicionados:</h6>
                                            <div className='d-flex align-items-center justify-content-between mb-3'>
                                                    <div className='d-flex align-items-center w-100 mb-2'>
                                                        <div className='bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2'><b>Nome:</b> 
                                                        
                                                        <div className='text-ellipsis '>
                                                            {valuesList.nome}
                                                        </div>
                                                        </div>
                                                        {limit && (
                                                            <div className='bg-secondary bg-opacity-25 text-ellipsis rounded fs-8 px-2 py-1 me-2'><b>Limite:</b> {valuesList.limite}</div>
                                                        )}
                                                    </div>
                                                    {preview &&
                                                        <div className='d-flex'>
                                                            <div className='d-flex flex-column me-2'>
                                                                <span className='bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2 mb-2'><b>Largura:</b> {valuesList.width}</span>
                                                                <span className='bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2'><b>Posição X:</b> {valuesList.x}</span>
                                                            </div>
                                                            <div className='d-flex'>
                                                                <div className='d-flex flex-column me-5'>
                                                                    <span className='bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2 mb-2'><b>Altura:</b> {valuesList.height}</span>
                                                                    <span className='bg-secondary bg-opacity-25 rounded fs-8 px-2 py-1 me-2'><b>Posição Y:</b> {valuesList.y}</span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    }
                                                <div>
                                                    <button className='d-flex align-items-center btn btn-danger' onClick={e => setValuesList(undefined)}>
                                                        <Image src='/svg/trash.svg' className='' width={16} height={16} alt='' />
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                </div>

                            </div>
                        </div>
                    </Col>
                    <Col lg={5}>
                        <ProductCard id={id} name={name} image={image} />
                        {/* crop */}
                        {preview && (
                            <div>
                                <ImageUploading
                                    multiple
                                    value={images}
                                    onChange={onImageChange}
                                    maxNumber={1}
                                >
                                    {({
                                        imageList,
                                        onImageUpload,
                                        onImageRemove,
                                        isDragging,
                                        dragProps,
                                    }) => (
                                        // write your building UI
                                        <div className="w-100 d-flex align-items-center justify-content-center">
                                            <div className='d-flex align-items-center position-relative bg-secondary bg-opacity-25 rounded-4' style={{ minWidth: '200px', minHeight: '100px' }}>
                                                {!imageList[0] && (
                                                    <div
                                                        className='text-center fs-7 p-4'
                                                        onClick={onImageUpload}
                                                        style={isDragging ? { color: 'red' } : undefined}
                                                        {...dragProps}
                                                    >
                                                        Clique aqui ou arraste e solte uma imagem
                                                    </div>
                                                )}
                                                {imageList[0] && (
                                                    <ReactCrop crop={crop} onChange={(c, p) => setCrop(p)}>
                                                        <img className='rounded-4' src={imageList[0].dataURL}  style={{ objectFit: 'contain' }} alt='' />
                                                    </ReactCrop>
                                                )}
                                                <button className='btn btn-danger position-absolute top-0 right-0 m-2 px-1 py-0' onClick={() => onImageRemove(0)}>
                                                    <Image src='/svg/trash.svg' width={16} height={16} alt='' />
                                                </button>
                                            </div>
                                        </div>
                                    )}
                                </ImageUploading>
                            </div>
                        )}
                        <div className='d-flex justify-content-end mt-3'>
                            <button onClick={e => sendVariants()} className='d-flex align-items-center btn bg-success text-light me-2'>Salvar
                                <Image src='/svg/save.svg' className='ms-2' width={20} height={20} alt='' />
                            </button>
                            <button onClick={handleBack} className="btn btn-dark">Voltar</button>
                        </div>
                    </Col>
                </Row>
            </div>
            <ToastContainer />
        </div>
    )
}

export default SimpleText