import TabElement from '@/interfaces/TabElement'
import api from '@/utils/api'
import React, { useEffect, useState } from 'react'
import { Row, Modal, Tab, Tabs, Card, Col } from 'react-bootstrap'
import Pagination from '../Pagination/Pagination'
import ProductTab from '@/interfaces/ProductTab'
import Variant from '@/interfaces/Variant'
import Image from 'next/image';

type TabVariantsModalProps = {
    show: boolean,
    onHide: () => void
    id: number
}

const TabVariantsModal: React.FC<TabVariantsModalProps> = ({ show, onHide, id }) => {

    const [tabs, setTabs] = useState<ProductTab[]>([])
    const [variants, setVariants] = useState<Variant[]>([])

    const [currentPage, setCurrentPage] = useState<number>(1);
    const [currentPageVariants, setCurrentPageVariants] = useState<number>(1);

    const [pagesOnApi, setPagesOnApi] = useState<number>(0);
    const [variantPagesOnApi, setVariantPagesOnApi] = useState<number>(0);

    const [offset, setOffset] = useState<number>(0);
    const [offsetVariants, setOffsetVariants] = useState<number>(0);

    const [limit, setLimit] = useState<number>(9);
    const [limitVariants, setLimitVariants] = useState<number>(9);


    // Fetch tabs
    useEffect(() => {
        const getTabs = async () => {
            try {
                const response = await api.get(`/tab?offset=${offset}&limit=${limit}&product_id=${id}`)
                setTabs(response.data[0])
                setPagesOnApi(response.data[1])
            } catch (error) {
                console.log(error)
            }
        }
        getTabs()
    }, [id, offset])

    const getVariants = async () => {
        try {
            const response = await api.get(`/variant/many?offset=${offsetVariants}&limit=${limitVariants}&productId=${id}`)
            setVariants(response.data[0])
            setVariantPagesOnApi(response.data[1])
            console.log(response.data)
            console.log(currentPageVariants)
            if (currentPageVariants > response.data[1]) {
                handlePageChangeVariants(response.data[1])
                console.log(response.data[1])
            }
        } catch (error) {
            console.log(error)
        }
    }

    // Fetch variants
    useEffect(() => {
        getVariants()
    }, [id, offsetVariants])



    const deleteVariant = async (id: number) => {
        const confirmed = window.confirm("Tem certeza que deseja remover esta variação?");
        if (confirmed) {
            try {
                const response = await api.delete(`/variant/${id}`);
                console.log(response);
                setVariants(variants.filter(variant => variant.id !== id));
                getVariants()
            } catch (error) {
                console.log(error);
            }
        }
    };

    const handlePageChange = (page: number) => {
        setCurrentPage(page);
        setOffset((page - 1) * limit);
    };

    const handlePageChangeVariants = (page: number) => {
        setCurrentPageVariants(page);
        setOffsetVariants((page - 1) * limitVariants);
    };

    return (
        <div>
            <Modal show={show} onHide={onHide} size="xl">
                <Modal.Header className='border-bottom-0' closeButton>
                    <h4>
                        Lista de Produtos
                    </h4>
                </Modal.Header>
                <Modal.Body>
                    <Tabs className='mb-3'>
                        <Tab eventKey="Descrições" title="Descrições">
                            <Row className='p-3'>
                                {tabs.map((tab, index) => (
                                    <>
                                        {tab.TabElements.map((element, index) => (
                                            <Col key={index} lg={4} className='mb-3'>
                                                <Card className='h-100 border-0 shadow'>
                                                    <Card.Body>
                                                        {/* Type */}
                                                        <Card.Title className='text-center fs-6 fw-normal text-capitalize opacity-75'>{element.tipo}</Card.Title>
                                                        {/* Tab name */}
                                                        <Card.Subtitle className="text-center fs-6 border-top border-bottom text-primary py-1 mb-3">{tab.titulo}</Card.Subtitle>
                                                        {/* Tab text */}
                                                        <Card.Text className="fs-8">
                                                            <div className="fw-bold mb-2">{element?.titulo}</div>
                                                            {/* Tab link */}
                                                            {element?.link &&
                                                                <Card.Link href={element?.link} className="text-decoration-none">Link</Card.Link>
                                                            }
                                                            <div>{element?.texto}</div>
                                                        </Card.Text>
                                                    </Card.Body>
                                                </Card>
                                            </Col>
                                        ))}
                                    </>
                                ))}
                            </Row>
                            <Pagination
                                currentPage={currentPage}
                                pagesOnApi={pagesOnApi}
                                onPageChange={handlePageChange}
                            />
                        </Tab>
                        <Tab eventKey="Customizações" title="Customizações">
                            <Row>
                                {variants.map((variant, index) => (
                                    <Col lg={4} key={index} className="mb-3">
                                        <Card className='h-100 border-0 shadow'>
                                            <div className='position-relative mb-2'>
                                                <button onClick={e => deleteVariant(variant.id)} className='btn btn-danger position-absolute end-1 top-1 py-1'>
                                                    <Image src='svg/trash.svg' width={16} height={16} alt='' />
                                                </button>
                                            </div>
                                            <Card.Body>
                                                {/* Type */}
                                                <Card.Title className='text-center fs-6 fw-normal text-capitalize opacity-75'>{variant.tipo}</Card.Title>
                                                {/* Tab name */}
                                                <Card.Subtitle className="text-center fs-6 border-top border-bottom text-primary py-1 mb-3">{variant.nome}</Card.Subtitle>
                                                {/* Tab text */}
                                                <Card.Text className="fs-8">
                                                    <div className="text-center mb-2">
                                                        <span className="fw-bold me-1">
                                                            Qtd de variações:
                                                        </span>
                                                        {variant?.VariantValues?.length}</div>
                                                    {variant.cor && <div className="text-center text-capitalize opacity-50 fw-bold mb-2">Cor</div>}
                                                    {variant.fonte && <div className="text-center text-capitalize opacity-50 fw-bold mb-2">Fonte</div>}
                                                    {variant.imagem && <div className="text-center text-capitalize opacity-50 fw-bold mb-2">Imagem</div>}
                                                    {variant.ocultoCart && <div className="text-center text-capitalize opacity-50 fw-bold mb-2">Oculto no carrinho</div>}
                                                    {variant.preview && <div className="text-center text-capitalize opacity-50 mb-2">Pré Visualização</div>}
                                                </Card.Text>
                                            </Card.Body>
                                        </Card>
                                    </Col>
                                ))}
                            </Row>

                            <Pagination currentPage={currentPageVariants} pagesOnApi={variantPagesOnApi} onPageChange={handlePageChangeVariants} />
                        </Tab>
                    </Tabs>
                </Modal.Body>

            </Modal>
        </div>
    )
}

export default TabVariantsModal